<?php
/***********************
 * CONFIGURATION
 ***********************/
require_once __DIR__ . '/db.php';

// SECURITY FIX: Use environment variables or constants from db.php
// Option 1: If you have them in db.php as constants
if (defined('STRIPE_SECRET_KEY') && defined('STRIPE_PUBLIC_KEY')) {
    $STRIPE_SECRET_KEY = STRIPE_SECRET_KEY;
    $STRIPE_PUBLIC_KEY = STRIPE_PUBLIC_KEY;
} 
// Option 2: Use environment variables (safer)
elseif (getenv('STRIPE_SECRET_KEY') && getenv('STRIPE_PUBLIC_KEY')) {
    $STRIPE_SECRET_KEY = getenv('STRIPE_SECRET_KEY');
    $STRIPE_PUBLIC_KEY = getenv('STRIPE_PUBLIC_KEY');
} 
// Option 3: Hardcoded as last resort (NOT RECOMMENDED FOR PRODUCTION)
else {
    // ROTATE THESE KEYS IMMEDIATELY - They were exposed in your previous code
    $STRIPE_SECRET_KEY = "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";
    $STRIPE_PUBLIC_KEY = "pk_live_51SWo0RPozIOd0PGPTy2DZypxJpFUwomxdPT5Cdk8pVtSyYkLAwtbHzPDI3KSbZCDjYW4Vk4spQlPG4BWt9k5b4HQ001Wa9O0A6";
}

// Validate that keys are set
if (empty($STRIPE_SECRET_KEY) || empty($STRIPE_PUBLIC_KEY)) {
    die("Payment system configuration error. Please contact support.");
}

// Get payment parameters from URL or POST
$student_id = (int)($_GET['student_id'] ?? $_POST['student_id'] ?? 0);
$source_table = $_GET['table'] ?? $_POST['table'] ?? '';
$package_id = (int)($_GET['package_id'] ?? $_POST['package_id'] ?? 0);
$payment_method = $_GET['payment_method'] ?? $_POST['payment_method'] ?? 'stripe';
$items_param = $_GET['items'] ?? $_POST['items'] ?? '';
$currency = $_GET['currency'] ?? $_POST['currency'] ?? 'EUR';

// Parse items from JSON string
$items = [];
if ($items_param) {
    $items = json_decode($items_param, true) ?: [];
}

// Calculate total amount
$total_amount = 0;
foreach ($items as $item_id => $amount) {
    $total_amount += (float)$amount;
}

// Convert to cents for Stripe
$AMOUNT = (int)($total_amount * 100);

// Get application details for description
$app_details = null;
if ($student_id > 0) {
    $stmt = $conn->prepare("SELECT first_name, last_name, email FROM student_applications WHERE id = ?");
    $stmt->bind_param('i', $student_id);
    $stmt->execute();
    $app_details = $stmt->get_result()->fetch_assoc();
    $stmt->close();
}

// Get package details
$package_details = null;
if ($package_id > 0) {
    $stmt = $conn->prepare("SELECT title FROM fee_packages WHERE id = ?");
    $stmt->bind_param('i', $package_id);
    $stmt->execute();
    $package_details = $stmt->get_result()->fetch_assoc();
    $stmt->close();
}

$description = "Payment for " . ($package_details['title'] ?? 'Package') . " - " . ($app_details['first_name'] ?? 'Student') . " " . ($app_details['last_name'] ?? '');

// Handle payment recording (POST request)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_GET['action']) && $_GET['action'] === 'record') {
    header('Content-Type: application/json; charset=utf-8');
    
    $rawInput = file_get_contents('php://input');
    if ($rawInput === false || trim($rawInput) === '') {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Empty request body']);
        exit;
    }
    
    $data = json_decode($rawInput, true);
    if (!is_array($data)) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Invalid JSON payload']);
        exit;
    }
    
    // Sanitize input
    $applicationId = (int) ($data['student_id'] ?? 0);
    $sourceTable   = (string) ($data['table'] ?? $source_table);
    $packageId     = (int) ($data['package_id'] ?? $package_id);
    $method        = trim((string) ($data['payment_method'] ?? 'stripe'));
    $comment       = trim((string) ($data['comment'] ?? ''));
    $items         = $data['items'] ?? [];
    
    // Validation
    $allowedTables = [
        'student_applications',
        'malta_applications',
        'turkey_applications'
    ];
    
    if (
        $applicationId <= 0 ||
        $packageId <= 0 ||
        $method === '' ||
        !in_array($sourceTable, $allowedTables, true) ||
        !is_array($items) ||
        empty($items)
    ) {
        http_response_code(422);
        echo json_encode(['success' => false, 'message' => 'Invalid or missing required fields']);
        exit;
    }
    
    // Start transaction
    $conn->begin_transaction();
    
    try {
        // Ensure package assignment
        $stmt = $conn->prepare(
            "SELECT id FROM application_packages
             WHERE application_id = ? AND source_table = ? AND package_id = ? LIMIT 1"
        );
        $stmt->bind_param('isi', $applicationId, $sourceTable, $packageId);
        $stmt->execute();
        $assigned = $stmt->get_result()->fetch_assoc();
        $stmt->close();
        
        if (!$assigned) {
            $stmt = $conn->prepare(
                "INSERT INTO application_packages
                 (application_id, source_table, package_id, assigned_at)
                 VALUES (?, ?, ?, NOW())"
            );
            $stmt->bind_param('isi', $applicationId, $sourceTable, $packageId);
            $stmt->execute();
            $stmt->close();
        }
        
        // Process item payments
        $totalRecorded = 0.0;
        $receiptItems  = [];
        
        foreach ($items as $feeItemId => $amount) {
            $feeItemId = (int) $feeItemId;
            $amount    = round((float) $amount, 2);
            
            if ($feeItemId <= 0 || $amount <= 0) {
                throw new RuntimeException('Invalid item payment data');
            }
            
            $stmt = $conn->prepare(
                "SELECT amount FROM fee_items WHERE id = ? AND package_id = ? LIMIT 1"
            );
            $stmt->bind_param('ii', $feeItemId, $packageId);
            $stmt->execute();
            $item = $stmt->get_result()->fetch_assoc();
            $stmt->close();
            
            if (!$item) {
                throw new RuntimeException('Fee item not found');
            }
            
            $stmt = $conn->prepare(
                "SELECT COALESCE(SUM(amount_paid),0)
                 FROM application_payments
                 WHERE application_id = ? AND source_table = ? AND fee_item_id = ? AND status = 'PAID'"
            );
            $stmt->bind_param('isi', $applicationId, $sourceTable, $feeItemId);
            $stmt->execute();
            $stmt->bind_result($alreadyPaid);
            $stmt->fetch();
            $stmt->close();
            
            if (($alreadyPaid + $amount) > (float) $item['amount']) {
                throw new RuntimeException('Overpayment detected');
            }
            
            $stmt = $conn->prepare(
                "INSERT INTO application_payments
                 (application_id, source_table, fee_item_id, amount_paid,
                  payment_method, payment_comment, status, paid_at)
                 VALUES (?, ?, ?, ?, ?, ?, 'PAID', NOW())"
            );
            $stmt->bind_param('isidss', $applicationId, $sourceTable, $feeItemId, $amount, $method, $comment);
            $stmt->execute();
            $stmt->close();
            
            $totalRecorded += $amount;
            
            $receiptItems[] = [
                'label'  => 'Item ' . $feeItemId,
                'amount' => $amount
            ];
        }
        
        // Package completion check
        $stmt = $conn->prepare(
            "SELECT SUM(fi.amount) AS expected, COALESCE(SUM(p.amount_paid),0) AS paid
             FROM fee_items fi
             LEFT JOIN application_payments p
               ON p.fee_item_id = fi.id
              AND p.application_id = ?
              AND p.status = 'PAID'
             WHERE fi.package_id = ?"
        );
        $stmt->bind_param('ii', $applicationId, $packageId);
        $stmt->execute();
        $totals = $stmt->get_result()->fetch_assoc();
        $stmt->close();
        
        if ((float)$totals['paid'] >= (float)$totals['expected']) {
            $stmt = $conn->prepare("UPDATE {$sourceTable} SET app_paid = 1 WHERE application_id = ?");
            $stmt->bind_param('i', $applicationId);
            $stmt->execute();
            $stmt->close();
        }
        
        // Receipt record
        $receiptNo = 'RCT-' . date('Ymd-His') . '-' . random_int(100, 999);
        
        $receiptHtml = generateReceiptHtml([
            'receipt_no' => $receiptNo,
            'student_id' => $applicationId,
            'items'      => $receiptItems,
            'total'      => $totalRecorded,
            'method'     => $method
        ]);
        
        $stmt = $conn->prepare(
            "INSERT INTO payment_receipts
             (receipt_no, application_id, source_table, package_id,
              total_amount, payment_method, receipt_html)
             VALUES (?, ?, ?, ?, ?, ?, ?)"
        );
        $stmt->bind_param('sisidss', $receiptNo, $applicationId, $sourceTable, $packageId, $totalRecorded, $method, $receiptHtml);
        $stmt->execute();
        $stmt->close();
        
        $conn->commit();
        
        // Send response
        echo json_encode([
            'success'     => true,
            'message'     => 'Payment recorded successfully',
            'receipt_no'  => $receiptNo,
            'total_paid'  => number_format($totalRecorded, 2, '.', ''),
            'items_count' => count($items)
        ]);
        
        // Async background tasks
        if (function_exists('fastcgi_finish_request')) {
            fastcgi_finish_request();
        }
        
        generateReceiptPdf($receiptHtml, $receiptNo);
        
        $emailUrl = 'https://xanderglobalscholars.com/sendReceiptEmail.php';
        $payload = http_build_query([
            'receipt_no' => $receiptNo,
            'secret'     => 'RCP_9fA8kKx_2026_SECURE'
        ]);
        
        $ch = curl_init($emailUrl);
        curl_setopt_array($ch, [
            CURLOPT_POST           => true,
            CURLOPT_POSTFIELDS     => $payload,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT        => 10,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_SSL_VERIFYHOST => false,
        ]);
        curl_exec($ch);
        curl_close($ch);
        
    } catch (Throwable $e) {
        $conn->rollback();
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'Payment failed',
            'error'   => $e->getMessage()
        ]);
        exit;
    }
    exit;
}

/***********************
 * CREATE PAYMENT INTENT
 ***********************/
$data = http_build_query([
    "amount" => $AMOUNT,
    "currency" => "eur", // SET TO EURO
    "payment_method_types[]" => "card",
    "description" => $description,
    "metadata[student_id]" => $student_id,
    "metadata[source_table]" => $source_table,
    "metadata[package_id]" => $package_id,
    "metadata[payment_method]" => $payment_method
]);

$ch = curl_init("https://api.stripe.com/v1/payment_intents");
curl_setopt($ch, CURLOPT_POST, true);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
curl_setopt($ch, CURLOPT_USERPWD, $STRIPE_SECRET_KEY . ":");
curl_setopt($ch, CURLOPT_TIMEOUT, 30); // Add timeout
curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true); // Ensure SSL verification

$response = curl_exec($ch);
if (curl_errno($ch)) {
    die("Connection error: " . curl_error($ch));
}
curl_close($ch);

$paymentIntent = json_decode($response, true);

if (!isset($paymentIntent['client_secret'])) {
    // Log the error for debugging but show user-friendly message
    error_log("Stripe API Error: " . $response);
    die("Payment gateway error. Please try again or contact support.");
}

$clientSecret = $paymentIntent['client_secret'];

/* =====================================================
   RECEIPT HTML GENERATOR
===================================================== */
function generateReceiptHtml(array $data): string
{
    ob_start(); ?>
<!DOCTYPE html>
<html>
<head>
<meta charset="UTF-8">
<title>Receipt</title>
<style>
@page { size: 80mm auto; margin: 0; }
body { width: 80mm; margin: 0; padding: 5mm; font-family: monospace; font-size: 12px; }
.center { text-align: center; }
.line { border-top: 1px dashed #000; margin: 6px 0; }
table { width: 100%; border-collapse: collapse; }
td { padding: 2px 0; }
.right { text-align: right; }
</style>
</head>
<body>
<div class="center"><strong>SCHOOL NAME</strong><br>OFFICIAL PAYMENT RECEIPT</div>
<div class="line"></div>
Receipt: <?= htmlspecialchars($data['receipt_no']) ?><br>
Student ID: <?= htmlspecialchars((string)$data['student_id']) ?><br>
Date: <?= date('Y-m-d H:i') ?><br>
<div class="line"></div>
<table>
<?php foreach ($data['items'] as $row): ?>
<tr><td><?= htmlspecialchars($row['label']) ?></td><td class="right"><?= number_format($row['amount'], 2) ?></td></tr>
<?php endforeach; ?>
</table>
<div class="line"></div>
<table><tr><td><strong>TOTAL</strong></td><td class="right"><strong><?= number_format($data['total'], 2) ?></strong></td></tr></table>
<div class="line"></div>
Payment: <?= htmlspecialchars($data['method']) ?><br>
<div class="center">Thank you<br>Keep this receipt</div>
</body>
</html>
<?php
    return ob_get_clean();
}
?>

<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <title>Secure Payment - Xander Global Scholars</title>
    <script src="https://js.stripe.com/v3/"></script>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --primary: #1e3a5f;
            --primary-dark: #0f2542;
            --accent: #ff8c42;
            --accent-dark: #e6732f;
            --bg: #f8fafc;
            --card: #ffffff;
            --text: #1e293b;
            --text-light: #64748b;
            --border: #e2e8f0;
            --shadow-lg: 0 10px 15px -3px rgba(0, 0, 0, 0.1), 0 4px 6px -2px rgba(0, 0, 0, 0.05);
            --transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
        }
        
        * {
            box-sizing: border-box;
            margin: 0;
            padding: 0;
        }
        
        body {
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif;
            background: var(--bg);
            color: var(--text);
            line-height: 1.6;
            -webkit-font-smoothing: antialiased;
        }
        
        .payment-container {
            max-width: 600px;
            margin: 40px auto;
            padding: 20px;
        }
        
        .payment-card {
            background: var(--card);
            border-radius: 20px;
            padding: 40px;
            box-shadow: var(--shadow-lg);
            border: 1px solid var(--border);
        }
        
        .payment-header {
            text-align: center;
            margin-bottom: 30px;
        }
        
        .payment-header h1 {
            color: var(--primary);
            font-size: 2rem;
            font-weight: 700;
            margin-bottom: 10px;
        }
        
        .payment-details {
            background: var(--bg);
            padding: 20px;
            border-radius: 12px;
            margin-bottom: 30px;
        }
        
        .detail-row {
            display: flex;
            justify-content: space-between;
            margin-bottom: 10px;
            font-size: 0.95rem;
        }
        
        .detail-row:last-child {
            border-top: 2px solid var(--border);
            padding-top: 15px;
            font-size: 1.2rem;
            font-weight: 700;
            color: var(--primary);
        }
        
        .card-element {
            border: 2px solid var(--border);
            padding: 16px;
            border-radius: 12px;
            background: white;
            transition: var(--transition);
        }
        
        .card-element:focus-within {
            border-color: var(--accent);
        }
        
        .payment-button {
            width: 100%;
            background: linear-gradient(135deg, var(--accent) 0%, var(--accent-dark) 100%);
            color: white;
            border: none;
            padding: 18px 24px;
            border-radius: 12px;
            font-weight: 600;
            font-size: 1.1rem;
            cursor: pointer;
            transition: var(--transition);
            margin-top: 20px;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 10px;
        }
        
        .payment-button:hover:not(:disabled) {
            transform: translateY(-2px);
            box-shadow: 0 6px 16px rgba(255, 140, 66, 0.3);
        }
        
        .payment-button:disabled {
            opacity: 0.6;
            cursor: not-allowed;
        }
        
        .result {
            margin-top: 20px;
            padding: 16px;
            border-radius: 8px;
            font-weight: 500;
            text-align: center;
        }
        
        .result.success {
            background: #10b981;
            color: white;
        }
        
        .result.error {
            background: #ef4444;
            color: white;
        }
        
        .back-link {
            text-align: center;
            margin-top: 20px;
        }
        
        .back-link a {
            color: var(--text-light);
            text-decoration: none;
            font-weight: 500;
        }
        
        .back-link a:hover {
            color: var(--primary);
        }
        
        .loading {
            display: inline-block;
            width: 20px;
            height: 20px;
            border: 3px solid rgba(255, 255, 255, 0.3);
            border-radius: 50%;
            border-top-color: white;
            animation: spin 1s ease-in-out infinite;
        }
        
        @keyframes spin {
            to { transform: rotate(360deg); }
        }
        
        .currency-note {
            text-align: center;
            color: var(--text-light);
            font-size: 0.9rem;
            margin-top: 5px;
        }
    </style>
</head>
<body>

<div class="payment-container">
    <div class="payment-card">
        <div class="payment-header">
            <h1>Secure Payment</h1>
            <p>Complete your payment securely with Stripe</p>
            <div class="currency-note">
                <i class="fas fa-euro-sign"></i> All amounts are in Euros (EUR)
            </div>
        </div>
        
        <?php if ($app_details && $package_details): ?>
        <div class="payment-details">
            <div class="detail-row">
                <span>Student:</span>
                <strong><?= htmlspecialchars($app_details['first_name'] . ' ' . $app_details['last_name']) ?></strong>
            </div>
            <div class="detail-row">
                <span>Email:</span>
                <strong><?= htmlspecialchars($app_details['email']) ?></strong>
            </div>
            <div class="detail-row">
                <span>Package:</span>
                <strong><?= htmlspecialchars($package_details['title']) ?></strong>
            </div>
            <div class="detail-row">
                <span>Application ID:</span>
                <strong>#<?= $student_id ?></strong>
            </div>
            <div class="detail-row">
                <span>Total Amount:</span>
                <strong>€<?= number_format($total_amount, 2) ?> EUR</strong>
            </div>
        </div>
        <?php endif; ?>
        
        <form id="payment-form">
            <div id="card-element" class="card-element"></div>
            <button id="payBtn" class="payment-button" type="submit">
                <i class="fas fa-lock"></i>
                <span>Pay €<?= number_format($total_amount, 2) ?> EUR</span>
            </button>
        </form>
        
        <div id="result" class="result" style="display: none;"></div>
        
        <div class="back-link">
            <a href="payment.php">
                <i class="fas fa-arrow-left"></i> Back to Payment Portal
            </a>
        </div>
    </div>
</div>

<script>
const stripe = Stripe("<?= $STRIPE_PUBLIC_KEY ?>");
const elements = stripe.elements();
const card = elements.create("card", {
    style: {
        base: {
            fontSize: '16px',
            color: '#1e293b',
            '::placeholder': {
                color: '#64748b'
            }
        }
    }
});
card.mount("#card-element");

const paymentForm = document.getElementById("payment-form");
const payBtn = document.getElementById("payBtn");
const resultDiv = document.getElementById("result");

paymentForm.onsubmit = async (e) => {
    e.preventDefault();
    
    // Show loading state
    payBtn.disabled = true;
    payBtn.innerHTML = '<div class="loading"></div> Processing...';
    resultDiv.style.display = 'none';
    
    try {
        const { paymentIntent, error } = await stripe.confirmCardPayment(
            "<?= $clientSecret ?>",
            { 
                payment_method: { 
                    card: card,
                    billing_details: {
                        name: "<?= htmlspecialchars($app_details['first_name'] . ' ' . $app_details['last_name'] ?? '') ?>",
                        email: "<?= htmlspecialchars($app_details['email'] ?? '') ?>"
                    }
                } 
            }
        );

        if (error) {
            throw error;
        }

        // Payment successful - record it
        await recordPayment(paymentIntent);
        
        // Show success message
        resultDiv.className = 'result success';
        resultDiv.textContent = `✅ Payment Successful! Transaction ID: ${paymentIntent.id}`;
        resultDiv.style.display = 'block';
        
        // Redirect after delay
        setTimeout(() => {
            window.location.href = 'payment.php?success=1&payment_id=' + encodeURIComponent(paymentIntent.id);
        }, 3000);
        
    } catch (error) {
        resultDiv.className = 'result error';
        resultDiv.textContent = '❌ ' + error.message;
        resultDiv.style.display = 'block';
        
        // Reset button
        payBtn.disabled = false;
        payBtn.innerHTML = `<i class="fas fa-lock"></i> <span>Pay €<?= number_format($total_amount, 2) ?> EUR</span>`;
    }
};

async function recordPayment(paymentIntent) {
    const items = <?= json_encode($items) ?>;
    
    const response = await fetch('stripe-payment.php?action=record', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify({
            student_id: <?= $student_id ?>,
            package_id: <?= $package_id ?>,
            payment_method: 'stripe',
            comment: `Stripe Payment ID: ${paymentIntent.id}`,
            items: items,
            stripe_payment_intent_id: paymentIntent.id,
            amount: <?= $total_amount ?>,
            currency: '<?= $currency ?>'
        })
    });
    
    if (!response.ok) {
        console.error('Failed to record payment:', await response.text());
    }
}
</script>

</body>
</html>