<?php
require_once __DIR__ . '/db.php';

echo "<h2>Database Setup for Xander Global Scholars Payment System</h2>";

// Check existing tables
$result = $conn->query("SHOW TABLES");
$existing_tables = [];
while ($row = $result->fetch_array()) {
    $existing_tables[] = $row[0];
}

echo "<h3>Existing Tables:</h3><ul>";
foreach ($existing_tables as $table) {
    echo "<li>$table</li>";
}
echo "</ul>";

// Tables to create - matching exactly what record-payment.php and student-manage.php expect
$tables_to_create = [
    'packages' => "
        CREATE TABLE IF NOT EXISTS packages (
            id INT(11) PRIMARY KEY AUTO_INCREMENT,
            code VARCHAR(20) UNIQUE,
            title VARCHAR(255) NOT NULL,
            currency VARCHAR(10) NOT NULL DEFAULT 'USD',
            total_expected DECIMAL(12,2),
            total_amount DECIMAL(10,2),
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        ) ENGINE=InnoDB DEFAULT CHARSET=latin1 COLLATE=latin1_swedish_ci;
    ",
    
    'payment_packages' => "
        CREATE TABLE IF NOT EXISTS payment_packages (
            id INT(11) PRIMARY KEY AUTO_INCREMENT,
            package_id INT(11) NOT NULL,
            name VARCHAR(255) NOT NULL,
            amount DECIMAL(12,2) NOT NULL,
            currency VARCHAR(10) NOT NULL DEFAULT 'USD',
            refundable TINYINT(1) DEFAULT 0,
            trigger_stage VARCHAR(100),
            is_refundable TINYINT(1) DEFAULT 0,
            payable_stage VARCHAR(100),
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        ) ENGINE=InnoDB DEFAULT CHARSET=latin1 COLLATE=latin1_swedish_ci;
    ",
    
    'student_assignments' => "
        CREATE TABLE IF NOT EXISTS student_assignments (
            id INT(11) PRIMARY KEY AUTO_INCREMENT,
            application_id INT(11) NOT NULL,
            source_table VARCHAR(50) NOT NULL,
            package_id INT(11) NOT NULL,
            assigned_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        ) ENGINE=InnoDB DEFAULT CHARSET=latin1 COLLATE=latin1_swedish_ci;
    ",
    
    'fee_items' => "
        CREATE TABLE IF NOT EXISTS fee_items (
            id INT(11) PRIMARY KEY AUTO_INCREMENT,
            package_id INT(11) NOT NULL,
            name VARCHAR(255) NOT NULL,
            amount DECIMAL(12,2) NOT NULL,
            currency VARCHAR(10) NOT NULL DEFAULT 'USD',
            refundable TINYINT(1) DEFAULT 0,
            trigger_stage VARCHAR(100),
            is_refundable TINYINT(1) DEFAULT 0,
            payable_stage VARCHAR(100),
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        ) ENGINE=InnoDB DEFAULT CHARSET=latin1 COLLATE=latin1_swedish_ci;
    ",
    
    'application_payments' => "
        CREATE TABLE IF NOT EXISTS application_payments (
            id INT(11) PRIMARY KEY AUTO_INCREMENT,
            application_id INT(11) NOT NULL,
            source_table VARCHAR(50) NOT NULL,
            fee_item_id INT(11),
            amount_paid DECIMAL(12,2) NOT NULL,
            payment_method VARCHAR(50) NOT NULL,
            payment_comment TEXT,
            status ENUM('PENDING', 'PAID', 'FAILED') DEFAULT 'PAID',
            paid_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            stripe_payment_intent_id VARCHAR(255)
        ) ENGINE=InnoDB DEFAULT CHARSET=latin1 COLLATE=latin1_swedish_ci;
    ",
    
    'application_packages' => "
        CREATE TABLE IF NOT EXISTS application_packages (
            id INT(11) PRIMARY KEY AUTO_INCREMENT,
            application_id INT(11) NOT NULL,
            source_table VARCHAR(50) NOT NULL,
            package_id INT(11) NOT NULL,
            assigned_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            UNIQUE KEY unique_assignment (application_id, source_table, package_id)
        ) ENGINE=InnoDB DEFAULT CHARSET=latin1 COLLATE=latin1_swedish_ci;
    ",
    
    'payment_receipts' => "
        CREATE TABLE IF NOT EXISTS payment_receipts (
            id INT(11) PRIMARY KEY AUTO_INCREMENT,
            receipt_no VARCHAR(50) UNIQUE NOT NULL,
            application_id INT(11) NOT NULL,
            source_table VARCHAR(50) NOT NULL,
            package_id INT(11) NOT NULL,
            total_amount DECIMAL(12,2) NOT NULL,
            payment_method VARCHAR(50) NOT NULL,
            receipt_html LONGTEXT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        ) ENGINE=InnoDB DEFAULT CHARSET=latin1 COLLATE=latin1_swedish_ci;
    "
];

// Create tables
echo "<h3>Creating Tables:</h3>";
foreach ($tables_to_create as $table_name => $sql) {
    if (!in_array($table_name, $existing_tables)) {
        try {
            $conn->query($sql);
            echo "<p style='color: green;'>✅ Table '$table_name' created successfully</p>";
        } catch (Exception $e) {
            echo "<p style='color: red;'>❌ Error creating table '$table_name': " . $e->getMessage() . "</p>";
        }
    } else {
        echo "<p style='color: orange;'>⚠️ Table '$table_name' already exists</p>";
    }
}

// Insert sample data if tables are empty
echo "<h3>Inserting Sample Data:</h3>";

// Check if packages table is empty
$result = $conn->query("SELECT COUNT(*) as count FROM packages");
$count = $result->fetch_assoc()['count'];

if ($count == 0) {
    // Insert sample packages
    $sample_packages = [
        ["STD001", "Student Visa Package", "USD", 2500.00, 2800.00],
        ["MAL001", "Malta Study Package", "EUR", 3500.00, 3800.00],
        ["TRK001", "Turkey Work Package", "USD", 2000.00, 2200.00],
        ["ADV001", "Advanced Processing", "USD", 1500.00, 1500.00]
    ];
    
    foreach ($sample_packages as $pkg) {
        $stmt = $conn->prepare("INSERT INTO packages (code, title, currency, total_expected, total_amount) VALUES (?, ?, ?, ?, ?)");
        $stmt->bind_param("sssdd", $pkg[0], $pkg[1], $pkg[2], $pkg[3], $pkg[4]);
        $stmt->execute();
        $stmt->close();
    }
    echo "<p style='color: green;'>✅ Sample packages inserted</p>";
    
    // Get package IDs for fee items
    $packages_result = $conn->query("SELECT id, title FROM packages");
    while ($pkg = $packages_result->fetch_assoc()) {
        $package_id = $pkg['id'];
        
        // Insert fee items for each package in payment_packages table
        $fee_items = [
            ["Application Fee", 500.00, "initial", "application"],
            ["Processing Fee", 300.00, "processing", "processing"],
            ["Visa Fee", 800.00, "visa", "visa"],
            ["Service Charge", 400.00, "final", "completion"]
        ];
        
        foreach ($fee_items as $fee) {
            $stmt = $conn->prepare("INSERT INTO payment_packages (package_id, name, amount, currency, refundable, trigger_stage, payable_stage) VALUES (?, ?, ?, 'USD', 1, ?, ?)");
            $stmt->bind_param("isds", $package_id, $fee[0], $fee[1], $fee[2], $fee[3]);
            $stmt->execute();
            $stmt->close();
        }
        
        // Also insert into fee_items table for compatibility
        foreach ($fee_items as $fee) {
            $stmt = $conn->prepare("INSERT INTO fee_items (package_id, name, amount, currency, refundable, trigger_stage, payable_stage) VALUES (?, ?, ?, 'USD', 1, ?, ?)");
            $stmt->bind_param("isds", $package_id, $fee[0], $fee[1], $fee[2], $fee[3]);
            $stmt->execute();
            $stmt->close();
        }
    }
    echo "<p style='color: green;'>✅ Sample fee items inserted into both payment_packages and fee_items tables</p>";
} else {
    echo "<p style='color: orange;'>⚠️ Packages table already has data</p>";
}

echo "<h3>Setup Complete!</h3>";
echo "<p><strong>Tables created match the exact structure expected by:</strong></p>";
echo "<ul>";
echo "<li><strong>record-payment.php</strong> - Uses fee_items, application_payments, application_packages, payment_receipts</li>";
echo "<li><strong>student-manage.php</strong> - Uses packages, payment_packages, student_assignments</li>";
echo "</ul>";
echo "<p><a href='payment.php' style='background: #1e3a5f; color: white; padding: 10px 20px; text-decoration: none; border-radius: 5px;'>Go to Payment Portal</a></p>";
?>
