<?php
require_once __DIR__ . '/db.php';

// Initialize variables
$selected_student = null;
$registration_error = $_GET['error'] ?? null;
$registered_new = $_GET['registered'] ?? null;
$success_message = $_GET['success'] ?? null;

// Get selected student data
if (isset($_GET['student_id'])) {
    $student_id = (int)$_GET['student_id'];
    $stmt = $conn->prepare("SELECT * FROM student_applications WHERE id = ?");
    $stmt->bind_param('i', $student_id);
    $stmt->execute();
    $selected_student = $stmt->get_result()->fetch_assoc();
    $stmt->close();
}

// Handle form submission for student registration
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['register_student'])) {
    $first_name = trim($_POST['first_name'] ?? '');
    $last_name = trim($_POST['last_name'] ?? '');
    $email = trim($_POST['email'] ?? '');
    $phone_number = trim($_POST['phone_number'] ?? '');
    $area_code = trim($_POST['area_code'] ?? '');
    
    if ($first_name && $last_name && $email && $phone_number) {
        // Check if email already exists
        $stmt = $conn->prepare("SELECT id FROM student_applications WHERE email = ?");
        $stmt->bind_param('s', $email);
        $stmt->execute();
        $existing = $stmt->get_result()->fetch_assoc();
        $stmt->close();
        
        if ($existing) {
            header("Location: payment.php?error=" . urlencode('Email already registered. Please search for your existing account.'));
            exit();
        }
        
        // Insert new student
        $stmt = $conn->prepare("INSERT INTO student_applications (user_id, first_name, last_name, email, area_code, phone_number) VALUES (?, ?, ?, ?, ?, ?)");
        $user_id = 'STU_' . time() . '_' . rand(1000, 9999);
        $stmt->bind_param('ssssss', $user_id, $first_name, $last_name, $email, $area_code, $phone_number);
        
        if ($stmt->execute()) {
            $student_id = $stmt->insert_id;
            $stmt->close();
            header("Location: payment.php?student_id=$student_id&registered=1");
            exit();
        } else {
            header("Location: payment.php?error=" . urlencode('Registration failed. Please try again.'));
            exit();
        }
    } else {
        header("Location: payment.php?error=" . urlencode('All fields are required.'));
        exit();
    }
}

// Handle form submission for student selection
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['select_student'])) {
    $student_id = (int)($_POST['student_id'] ?? 0);
    
    if ($student_id > 0) {
        header("Location: payment.php?student_id=$student_id");
        exit();
    }
}

// Handle form submission for payment
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['process_payment'])) {
    $student_id = (int)($_GET['student_id'] ?? 0);
    
    if ($student_id === 0) {
        header("Location: payment.php?error=" . urlencode('Please select or register a student first.'));
        exit();
    }
    
    $package_id = (int)($_POST['package_id'] ?? 0);
    $selected_items = $_POST['selected_items'] ?? [];
    $item_amounts = $_POST['item_amount'] ?? [];
    $payment_method = $_POST['payment_method'] ?? 'stripe';
    
    if ($package_id > 0 && !empty($selected_items)) {
        // Prepare items data with custom amounts
        $items_data = [];
        $total = 0;
        
        foreach ($selected_items as $item_id) {
            $amount = isset($item_amounts[$item_id]) ? (float)$item_amounts[$item_id] : 0;
            if ($amount > 0) {
                $items_data[$item_id] = $amount;
                $total += $amount;
            }
        }
        
        if ($total > 0) {
            $items_json = json_encode($items_data);
            // Get currency from selected package
            $package_currency = $packages_summary[$package_id]['currency'] ?? 'USD';
            header("Location: stripe-payment.php?student_id=$student_id&package_id=$package_id&payment_method=$payment_method&currency=$package_currency&items=" . urlencode($items_json));
            exit();
        } else {
            header("Location: payment.php?student_id=$student_id&error=" . urlencode('Please select at least one item with amount greater than 0.'));
            exit();
        }
    } else {
        header("Location: payment.php?student_id=$student_id&error=" . urlencode('Please select a package and at least one item.'));
        exit();
    }
}

// Fetch packages from database - using total_amount column
$packages_result = $conn->query("SELECT * FROM fee_packages ORDER BY title ASC");

// Fetch fee items for each package
$fee_items = [];
$fee_items_result = $conn->query("SELECT fi.*, p.title as package_title, p.currency as package_currency, p.total_amount as package_total FROM fee_items fi LEFT JOIN fee_packages p ON fi.package_id = p.id ORDER BY fi.package_id, fi.id");
while ($item = $fee_items_result->fetch_assoc()) {
    if (!isset($fee_items[$item['package_id']])) {
        $fee_items[$item['package_id']] = [];
    }
    $fee_items[$item['package_id']][] = $item;
}

// Create packages summary - using total_amount
$packages_summary = [];
$packages_result->data_seek(0);
while ($package = $packages_result->fetch_assoc()) {
    $packages_summary[$package['id']] = [
        'title' => $package['title'],
        'currency' => $package['currency'],
        'total_expected' => $package['total_expected'],
        'total_amount' => $package['total_amount'],
        'payment_items' => $fee_items[$package['id']] ?? []
    ];
}

$pageTitle = 'Secure Payment Portal - Xander Global Scholars';

?>

<div class="page-hero">
    <div class="hero-content">
        <h1><i class="fas fa-shield-alt"></i> Secure Payment Portal</h1>
        <p class="hero-subtitle">Complete your payment in three simple steps</p>
        <div class="progress-steps">
            <div class="step <?= $selected_student ? 'completed active' : '' ?>">
                <div class="step-icon">1</div>
                <div class="step-text">Student Verification</div>
            </div>
            <div class="step <?= $selected_student ? 'active' : '' ?>">
                <div class="step-icon">2</div>
                <div class="step-text">Package Selection</div>
            </div>
            <div class="step">
                <div class="step-icon">3</div>
                <div class="step-text">Payment Processing</div>
            </div>
        </div>
    </div>
</div>

<div class="page-content">
    <?php if ($success_message): ?>
        <div class="alert alert-success">
            <i class="fas fa-check-circle"></i>
            <div class="alert-content">
                <strong>Success!</strong> <?= htmlspecialchars($success_message) ?>
            </div>
            <button class="alert-close" onclick="this.parentElement.style.display='none'">
                <i class="fas fa-times"></i>
            </button>
        </div>
    <?php endif; ?>
    
    <?php if ($registration_error): ?>
        <div class="alert alert-error">
            <i class="fas fa-exclamation-circle"></i>
            <div class="alert-content">
                <strong>Attention Required!</strong> <?= htmlspecialchars($registration_error) ?>
            </div>
            <button class="alert-close" onclick="this.parentElement.style.display='none'">
                <i class="fas fa-times"></i>
            </button>
        </div>
    <?php endif; ?>
    
    <div class="dashboard-container">
        <?php if (!$selected_student): ?>
            <!-- Student Verification Section -->
            <div class="dashboard-card">
                <div class="card-header">
                    <h2><i class="fas fa-user-graduate"></i> Student Verification</h2>
                    <p class="card-subtitle">To proceed with your payment, please begin by entering your Email Address, Phone Number, or Full Name in the fields provided below.</p>
                </div>
                
                <div class="card-body">
                    <!-- Live Search Section -->
                    <div class="search-section">
                        <h3>
  <i class="fas fa-search"></i>
  Type Your Email, Phone Number, or Full Name to Proceed to the Payment Process
</h3>

<p class="section-description">
  Start typing to search for an existing account using your name, email, or phone number.
  If you are not registered, please continue by entering your email address to register.
</p>

                        <form method="POST" id="student-search-form" class="search-form">
                            <input type="hidden" name="select_student" value="1">
                            <input type="hidden" name="student_id" id="selected-student-id" value="">
                            
                            <div class="form-group">
                                <div class="search-container">
                                    <div class="input-with-icon">
                                        <i class="fas fa-search input-icon"></i>
                                        <input type="text" 
                                               id="search-input" 
                                               placeholder="Type student name, email, or phone..." 
                                               autocomplete="off"
                                               class="search-input">
                                    </div>
                                    <div id="search-results" class="search-results"></div>
                                </div>
                                <div class="input-hint">
                                    <i class="fas fa-info-circle"></i> Search by name, email, or phone number
                                </div>
                            </div>
                            
                            <div class="form-actions">
                                <button type="submit" class="btn btn-primary" id="select-btn" disabled>
                                    <i class="fas fa-user-check"></i> Select Student
                                </button>
                            </div>
                        </form>
                    </div>
                    
                    <!-- Registration Form (Auto-shown when no student found) -->
                    <div id="register-section" class="register-section" style="display: none;">
                        <div class="divider">
                            <span>OR</span>
                        </div>
                        
                        <div class="register-prompt">
                            <div class="prompt-icon">
                                <i class="fas fa-user-plus"></i>
                            </div>
                            <div class="prompt-content">
                                <h4>Register New Student</h4>
                                <p>Student not found? Register as a new student to continue.</p>
                            </div>
                        </div>
                        
                        <form method="POST" id="registration-form" class="registration-form">
                            <input type="hidden" name="register_student" value="1">
                            
                            <div class="form-grid">
                                <div class="form-group">
                                    <label for="first_name">
                                        <i class="fas fa-user"></i> First Name
                                        <span class="required">*</span>
                                    </label>
                                    <input type="text" 
                                           name="first_name" 
                                           id="first_name" 
                                           required 
                                           placeholder="Enter first name"
                                           class="form-control">
                                </div>
                                
                                <div class="form-group">
                                    <label for="last_name">
                                        <i class="fas fa-user"></i> Last Name
                                        <span class="required">*</span>
                                    </label>
                                    <input type="text" 
                                           name="last_name" 
                                           id="last_name" 
                                           required 
                                           placeholder="Enter last name"
                                           class="form-control">
                                </div>
                            </div>
                            
                            <div class="form-group">
                                <label for="email">
                                    <i class="fas fa-envelope"></i> Email Address
                                    <span class="required">*</span>
                                </label>
                                <input type="email" 
                                       name="email" 
                                       id="reg-email" 
                                       required 
                                       placeholder="student@example.com"
                                       class="form-control">
                            </div>
                            
                            <div class="form-grid">
                                <div class="form-group">
                                    <label for="area_code">
                                        <i class="fas fa-phone"></i> Area Code
                                    </label>
                                    <div class="input-group">
                                        <span class="input-group-text">+</span>
                                        <input type="text" 
                                               name="area_code" 
                                               id="area_code" 
                                               placeholder="1"
                                               maxlength="4"
                                               class="form-control">
                                    </div>
                                </div>
                                
                                <div class="form-group">
                                    <label for="phone_number">
                                        <i class="fas fa-mobile-alt"></i> Phone Number
                                        <span class="required">*</span>
                                    </label>
                                    <input type="tel" 
                                           name="phone_number" 
                                           id="phone_number" 
                                           required 
                                           placeholder="1234567890"
                                           class="form-control">
                                </div>
                            </div>
                            
                            <div class="form-actions">
                                <button type="submit" class="btn btn-success">
                                    <i class="fas fa-user-plus"></i> Register & Continue
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        <?php else: ?>
            <!-- Student Information & Package Selection -->
            <div class="dashboard-grid">
                <!-- Student Info Card -->
                <div class="student-card">
                    <div class="card-header">
                        <h3><i class="fas fa-user-circle"></i> Student Information</h3>
                        <a href="payment.php" class="btn-change">
                            <i class="fas fa-exchange-alt"></i> Change Student
                        </a>
                    </div>
                    
                    <div class="student-info">
                        <div class="student-avatar">
                            <i class="fas fa-user-graduate"></i>
                        </div>
                        
                        <div class="student-details">
                            <h4><?= htmlspecialchars($selected_student['first_name'] . ' ' . $selected_student['last_name']) ?></h4>
                            <div class="detail-item">
                                <i class="fas fa-envelope"></i>
                                <span><?= htmlspecialchars($selected_student['email']) ?></span>
                            </div>
                            <?php if ($selected_student['phone_number']): ?>
                                <div class="detail-item">
                                    <i class="fas fa-phone"></i>
                                    <span><?= htmlspecialchars($selected_student['area_code'] . ' ' . $selected_student['phone_number']) ?></span>
                                </div>
                            <?php endif; ?>
                            <div class="detail-item">
                                <i class="fas fa-id-card"></i>
                                <span>ID: <?= htmlspecialchars($selected_student['user_id']) ?></span>
                            </div>
                        </div>
                    </div>
                    
                    <?php if ($registered_new): ?>
                        <div class="registration-success">
                            <i class="fas fa-check-circle"></i>
                            <span>New student registered successfully!</span>
                        </div>
                    <?php endif; ?>
                </div>
                
                <!-- Package Selection Card -->
                <div class="package-card">
                    <div class="card-header">
                        <h3><i class="fas fa-box-open"></i> Package & Fee Selection</h3>
                        <p class="card-subtitle">Select your package and customize fee items</p>
                    </div>
                    
                    <div class="card-body">
                        <form method="POST" id="payment-form" class="payment-form">
                            <input type="hidden" name="process_payment" value="1">
                            
                            <!-- Package Selection -->
                            <div class="form-group">
                                <label for="package-select">
                                    <i class="fas fa-cube"></i> Select Package
                                    <span class="required">*</span>
                                </label>
                                <div class="select-wrapper">
                                    <select name="package_id" id="package-select" required onchange="loadFeeItems(this.value)">
                                        <option value="">-- Choose a package --</option>
                                        <?php foreach ($packages_summary as $package_id => $package): ?>
                                            <option value="<?= $package_id ?>">
                                                <?= htmlspecialchars($package['title']) ?> 
                                                - <?= $package['currency'] ?> <?= number_format($package['total_amount'], 2) ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                    <i class="fas fa-chevron-down select-icon"></i>
                                </div>
                                <small class="form-help">Choose the package that best fits your needs</small>
                            </div>
                            
                            <!-- Package Info (shown when package is selected) -->
                            <div id="package-info" style="display: none;" class="package-info-card">
                                <div class="package-info-header">
                                    <h4><i class="fas fa-info-circle"></i> Package Details</h4>
                                </div>
                                <div class="package-info-body">
                                    <div class="info-row">
                                        <span class="info-label">Package:</span>
                                        <span id="package-title" class="info-value"></span>
                                    </div>
                                    <div class="info-row">
                                        <span class="info-label">Currency:</span>
                                        <span id="package-currency" class="info-value"></span>
                                    </div>
                                    <div class="info-row">
                                        <span class="info-label">Package Total:</span>
                                        <span id="package-total" class="info-value amount"></span>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Fee Items Section -->
                            <div id="fee-items-section" style="display: none;">
                                <div class="section-header">
                                    <h4><i class="fas fa-list-alt"></i> Available Fee Items</h4>
                                    <p class="section-description">Select items and adjust amounts (cannot exceed original price)</p>
                                </div>
                                
                                <div id="fee-items-list" class="fee-items-grid"></div>
                                
                                <!-- Selected Items Summary -->
                                <div id="selected-summary" class="selected-summary" style="display: none;">
                                    <div class="summary-header">
                                        <h4><i class="fas fa-check-circle"></i> Selected Items</h4>
                                    </div>
                                    <div id="selected-items-list" class="selected-items-list"></div>
                                </div>
                                
                                <!-- Enhanced Payment Summary -->
                                <div class="payment-summary-enhanced">
                                    <div class="summary-header">
                                        <h3><i class="fas fa-calculator"></i> Payment Summary</h3>
                                        <div class="summary-badge">
                                            <i class="fas fa-shield-alt"></i>
                                            <span>Secure Checkout</span>
                                        </div>
                                    </div>
                                    
                                    <div class="summary-content">
                                        <div class="summary-section">
                                            <h4><i class="fas fa-shopping-cart"></i> Order Details</h4>
                                            <div class="items-list" id="selected-items-list">
                                                <!-- Items will be populated by JavaScript -->
                                            </div>
                                        </div>
                                        
                                        <div class="summary-section">
                                            <div class="total-amount-display">
                                                <h4><i class="fas fa-calculator"></i> Total Amount</h4>
                                                <div class="total-amount-value" id="total-amount">€0.00</div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                
                                <!-- Payment Method -->
                                <div class="payment-methods">
                                    <h4><i class="fas fa-credit-card"></i> Payment Method</h4>
                                    <div class="method-options">
                                        <label class="method-option active">
                                            <input type="radio" name="payment_method" value="stripe" checked>
                                            <div class="method-content">
                                                <div class="method-icon">
                                                    <i class="fab fa-cc-stripe"></i>
                                                </div>
                                                <div class="method-info">
                                                    <h5>Credit/Debit Card</h5>
                                                    <p>Pay securely with Stripe</p>
                                                </div>
                                            </div>
                                            <div class="method-check">
                                                <i class="fas fa-check"></i>
                                            </div>
                                        </label>
                                        
                                        <label class="method-option">
                                            <input type="radio" name="payment_method" value="mtn_momo">
                                            <div class="method-content">
                                                <div class="method-icon">
                                                    <i class="fas fa-mobile-alt"></i>
                                                </div>
                                                <div class="method-info">
                                                    <h5>MTN MOBILE MONEY Rwanda</h5>
                                                    <p>Mobile money payment</p>
                                                </div>
                                            </div>
                                            <div class="method-check">
                                                <i class="fas fa-check"></i>
                                            </div>
                                        </label>
                                    </div>
                                </div>
                                
                                <!-- Submit Button -->
                                <div class="form-actions">
                                    <button type="submit" class="btn btn-primary btn-pay" id="submit-btn" disabled>
                                        <i class="fas fa-lock"></i>
                                        <span>Proceed to Secure Payment</span>
                                        <i class="fas fa-arrow-right"></i>
                                    </button>
                                    <p class="security-note">
                                        <i class="fas fa-shield-alt"></i>
                                        Your payment is secured with 256-bit SSL encryption
                                    </p>
                                </div>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        <?php endif; ?>
        
        <
    </div>
</div>


<script>
// Fee items data for JavaScript
const feeItemsData = <?= json_encode($fee_items) ?>;
const packagesData = <?= json_encode($packages_summary) ?>;

// Live Search Functionality
let searchTimeout;
const searchInput = document.getElementById('search-input');
const searchResults = document.getElementById('search-results');
const selectBtn = document.getElementById('select-btn');
const selectedStudentId = document.getElementById('selected-student-id');
const registerSection = document.getElementById('register-section');

if (searchInput && searchResults) {
    searchInput.addEventListener('input', function() {
        clearTimeout(searchTimeout);
        const query = this.value.trim();
        
        if (query.length < 2) {
            searchResults.innerHTML = '';
            selectBtn.disabled = true;
            selectedStudentId.value = '';
            if (registerSection) {
                registerSection.style.display = 'none';
            }
            return;
        }
        
        searchTimeout = setTimeout(() => {
            searchResults.innerHTML = '<div class="search-loading"><i class="fas fa-spinner fa-spin"></i> Searching student database...</div>';
            
            fetch(`search-students.php?q=${encodeURIComponent(query)}`)
                .then(response => {
                    if (!response.ok) throw new Error('Network error');
                    return response.json();
                })
                .then(students => {
                    displaySearchResults(students);
                })
                .catch(error => {
                    console.error('Search error:', error);
                    searchResults.innerHTML = '<div class="search-no-results"><i class="fas fa-exclamation-circle"></i> Error loading results. Please try again.</div>';
                });
        }, 300);
    });
    
    document.addEventListener('click', function(e) {
        if (!e.target.closest('.search-container')) {
            searchResults.innerHTML = '';
        }
    });
}

function displaySearchResults(students) {
    if (!students || students.length === 0) {
        searchResults.innerHTML = '<div class="search-no-results"><i class="fas fa-search"></i> No students found matching your search.</div>';
        selectBtn.disabled = true;
        selectedStudentId.value = '';
        
        if (registerSection) {
            registerSection.style.display = 'block';
            setTimeout(() => {
                registerSection.scrollIntoView({ behavior: 'smooth', block: 'center' });
            }, 300);
        }
        return;
    }
    
    if (registerSection) {
        registerSection.style.display = 'none';
    }
    
    let html = '<div class="search-results-list">';
    students.forEach(student => {
        const fullName = `${student.first_name} ${student.last_name}`;
        const phone = student.area_code && student.phone_number ? 
            `${student.area_code} ${student.phone_number}` : 'No phone number';
        
        const safeName = escapeHtml(fullName);
        const safeEmail = escapeHtml(student.email);
        const safePhone = escapeHtml(phone);
        
        html += `
            <div class="search-result-item" onclick="selectStudent(${student.id}, '${safeEmail}', '${student.first_name}', '${student.last_name}')">
                <div class="result-avatar">
                    <i class="fas fa-user"></i>
                </div>
                <div class="result-info">
                    <div class="result-name">${safeName}</div>
                    <div class="result-email"><i class="fas fa-envelope"></i> ${safeEmail}</div>
                    <div class="result-phone"><i class="fas fa-phone"></i> ${safePhone}</div>
                    <div class="result-id"><i class="fas fa-id-card"></i> ${student.user_id}</div>
                </div>
                <div class="result-select">
                    <i class="fas fa-chevron-right"></i>
                </div>
            </div>
        `;
    });
    html += '</div>';
    
    searchResults.innerHTML = html;
}

function selectStudent(studentId, email, firstName, lastName) {
    searchInput.value = `${firstName} ${lastName} (${email})`;
    selectedStudentId.value = studentId;
    searchResults.innerHTML = '';
    selectBtn.disabled = false;
    
    selectBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Selecting...';
    selectBtn.disabled = true;
    
    const form = document.getElementById('student-search-form');
    setTimeout(() => {
        form.submit();
    }, 100);
}

function loadFeeItems(packageId) {
    const feeItemsSection = document.getElementById('fee-items-section');
    const feeItemsList = document.getElementById('fee-items-list');
    const submitBtn = document.getElementById('submit-btn');
    const packageSelect = document.getElementById('package-select');
    const packageInfo = document.getElementById('package-info');
    const selectedSummary = document.getElementById('selected-summary');
    
    if (!packageId) {
        feeItemsSection.style.display = 'none';
        packageInfo.style.display = 'none';
        selectedSummary.style.display = 'none';
        submitBtn.disabled = true;
        return;
    }
    
    const package = packagesData[packageId];
    if (!package) {
        feeItemsSection.style.display = 'none';
        packageInfo.style.display = 'none';
        selectedSummary.style.display = 'none';
        submitBtn.disabled = true;
        return;
    }
    
    // Show package info
    document.getElementById('package-title').textContent = package.title;
    document.getElementById('package-currency').textContent = package.currency;
    document.getElementById('package-total').textContent = `${package.currency} ${parseFloat(package.total_amount).toFixed(2)}`;
    packageInfo.style.display = 'block';
    
    const items = feeItemsData[packageId] || [];
    const currency = package.currency || 'USD';
    
    if (items.length === 0) {
        feeItemsList.innerHTML = '<div class="no-items"><i class="fas fa-box-open"></i> No fee items available for this package.</div>';
        feeItemsSection.style.display = 'block';
        selectedSummary.style.display = 'none';
        submitBtn.disabled = true;
        updateTotals(0, 0, currency);
        return;
    }
    
    let html = '';
    
    items.forEach((item) => {
        const amount = parseFloat(item.total_expected || item.amount || 0);
        const itemCurrency = item.package_currency || currency;
        const itemId = escapeHtml(item.id.toString());
        const itemTitle = escapeHtml(item.title || item.name);
        const itemCode = escapeHtml(item.code || 'N/A');
        const description = item.description ? escapeHtml(item.description) : '';
        
        html += `
            <div class="fee-item-card" data-original-amount="${amount}">
                <div class="fee-item-header">
                    <div class="custom-checkbox">
                        <input type="checkbox" 
                               id="fee_${itemId}" 
                               name="selected_items[]" 
                               value="${itemId}"
                               data-original-amount="${amount}"
                               data-currency="${itemCurrency}"
                               onchange="toggleFeeItem(this, ${itemId})">
                        <label for="fee_${itemId}">
                            <div class="checkbox-box">
                                <i class="fas fa-check"></i>
                            </div>
                        </label>
                    </div>
                    <div class="fee-item-info">
                        <h5>${itemTitle}</h5>
                        ${description ? `<p class="fee-description">${description}</p>` : ''}
                        <div class="fee-item-meta">
                            <span class="fee-code"><i class="fas fa-hashtag"></i> ${itemCode}</span>
                            ${item.refundable ? '<span class="refundable-badge"><i class="fas fa-undo"></i> Refundable</span>' : ''}
                        </div>
                    </div>
                </div>
                
                <div class="fee-item-amount">
                    <div class="amount-control">
                        <div class="amount-input-group">
                            <span class="currency-label">${itemCurrency}</span>
                            <input type="number" 
                                   step="0.01" 
                                   min="0" 
                                   max="${amount}"
                                   class="amount-input" 
                                   id="amount_${itemId}" 
                                   name="item_amount[${itemId}]" 
                                   value="${amount.toFixed(2)}"
                                   disabled
                                   oninput="updateAmount(${itemId}, this.value)"
                                   placeholder="0.00">
                        </div>
                        <div class="amount-hint">
                            <small>Max: ${itemCurrency} ${amount.toFixed(2)}</small>
                        </div>
                    </div>
                </div>
            </div>
        `;
    });
    
    feeItemsList.innerHTML = html;
    feeItemsSection.style.display = 'block';
    selectedSummary.style.display = 'none';
    
    setTimeout(() => {
        feeItemsSection.style.opacity = '1';
        feeItemsSection.style.transform = 'translateY(0)';
    }, 10);
    
    updateSelectedItemsList();
    calculateTotal();
}

function toggleFeeItem(checkbox, itemId) {
    const amountInput = document.getElementById(`amount_${itemId}`);
    const feeItemCard = amountInput.closest('.fee-item-card');
    const selectedSummary = document.getElementById('selected-summary');
    
    if (checkbox.checked) {
        amountInput.disabled = false;
        feeItemCard.classList.add('selected');
        selectedSummary.style.display = 'block';
    } else {
        amountInput.disabled = true;
        feeItemCard.classList.remove('selected');
        // Reset to original amount
        const originalAmount = checkbox.getAttribute('data-original-amount');
        amountInput.value = parseFloat(originalAmount).toFixed(2);
    }
    
    updateSelectedItemsList();
    calculateTotal();
}

function updateAmount(itemId, value) {
    const checkbox = document.getElementById(`fee_${itemId}`);
    const originalAmount = parseFloat(checkbox.getAttribute('data-original-amount'));
    const input = document.getElementById(`amount_${itemId}`);
    
    // Ensure value doesn't exceed original amount
    let newValue = parseFloat(value) || 0;
    if (newValue > originalAmount) {
        newValue = originalAmount;
        input.value = newValue.toFixed(2);
        showMessage('Cannot exceed original amount', 'warning');
    }
    
    if (newValue < 0) {
        newValue = 0;
        input.value = newValue.toFixed(2);
    }
    
    updateSelectedItemsList();
    calculateTotal();
}

function updateSelectedItemsList() {
    const checkboxes = document.querySelectorAll('input[name="selected_items[]"]:checked');
    const selectedList = document.getElementById('selected-items-list');
    const selectedSummary = document.getElementById('selected-summary');
    
    if (checkboxes.length === 0) {
        selectedSummary.style.display = 'none';
        return;
    }
    
    let html = '';
    checkboxes.forEach(checkbox => {
        const itemId = checkbox.value;
        const amountInput = document.getElementById(`amount_${itemId}`);
        const amount = parseFloat(amountInput.value) || 0;
        const currency = checkbox.getAttribute('data-currency') || '$';
        const feeItemCard = checkbox.closest('.fee-item-card');
        const itemTitle = feeItemCard.querySelector('.fee-item-info h5').textContent;
        
        html += `
            <div class="selected-item">
                <div class="selected-item-info">
                    <span class="selected-item-name">${itemTitle}</span>
                    <span class="selected-item-amount">${currency} ${amount.toFixed(2)}</span>
                </div>
            </div>
        `;
    });
    
    selectedList.innerHTML = html;
    selectedSummary.style.display = 'block';
}

function calculateTotal() {
    const checkboxes = document.querySelectorAll('input[name="selected_items[]"]:checked');
    let total = 0;
    
    checkboxes.forEach(checkbox => {
        const itemId = checkbox.value;
        const amountInput = document.getElementById(`amount_${itemId}`);
        const amount = parseFloat(amountInput.value) || 0;
        total += amount;
    });
    
    updateTotal(total);
}

function updateTotal(total) {
    let currency = '$';
    const firstChecked = document.querySelector('input[name="selected_items[]"]:checked');
    if (firstChecked) {
        currency = firstChecked.getAttribute('data-currency') || '$';
    }
    
    document.getElementById('total-amount').textContent = `${currency} ${total.toFixed(2)}`;
    
    const submitBtn = document.getElementById('submit-btn');
    submitBtn.disabled = total === 0;
    
    if (total > 0) {
        submitBtn.querySelector('span').textContent = `Pay ${currency} ${total.toFixed(2)}`;
    } else {
        submitBtn.querySelector('span').textContent = 'Proceed to Secure Payment';
    }
}

function showMessage(message, type) {
    const messageDiv = document.createElement('div');
    messageDiv.className = `alert alert-${type}`;
    messageDiv.innerHTML = `
        <i class="fas fa-${type === 'warning' ? 'exclamation-triangle' : 'info-circle'}"></i>
        <div class="alert-content">${message}</div>
        <button class="alert-close" onclick="this.parentElement.remove()">
            <i class="fas fa-times"></i>
        </button>
    `;
    
    const pageContent = document.querySelector('.page-content');
    pageContent.insertBefore(messageDiv, pageContent.firstChild);
    
    setTimeout(() => {
        if (messageDiv.parentNode) {
            messageDiv.remove();
        }
    }, 5000);
}

function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

// Form validation and initialization
document.addEventListener('DOMContentLoaded', function() {
    // Payment method selection
    const methodOptions = document.querySelectorAll('.method-option');
    methodOptions.forEach(option => {
        option.addEventListener('click', function() {
            methodOptions.forEach(opt => opt.classList.remove('active'));
            this.classList.add('active');
        });
    });
    
    // Initialize first method as selected
    if (methodOptions.length > 0) {
        methodOptions[0].classList.add('active');
    }
    
    // Form validation for registration
    const registrationForm = document.getElementById('registration-form');
    if (registrationForm) {
        registrationForm.addEventListener('submit', function(e) {
            const phoneInput = document.getElementById('phone_number');
            const phoneValue = phoneInput.value.trim();
            
            if (!/^\d{10,15}$/.test(phoneValue)) {
                e.preventDefault();
                showMessage('Please enter a valid phone number (10-15 digits, numbers only).', 'warning');
                phoneInput.focus();
                return false;
            }
            
            const areaCodeInput = document.getElementById('area_code');
            const areaCodeValue = areaCodeInput.value.trim();
            if (areaCodeValue && !/^\d{1,4}$/.test(areaCodeValue)) {
                e.preventDefault();
                showMessage('Please enter a valid area code (1-4 digits).', 'warning');
                areaCodeInput.focus();
                return false;
            }
            
            return true;
        });
    }
    
    // Phone number formatting
    const phoneInput = document.getElementById('phone_number');
    if (phoneInput) {
        phoneInput.addEventListener('input', function(e) {
            let value = e.target.value.replace(/\D/g, '');
            if (value.length > 15) value = value.substr(0, 15);
            e.target.value = value;
        });
    }
    
    // Area code formatting
    const areaCodeInput = document.getElementById('area_code');
    if (areaCodeInput) {
        areaCodeInput.addEventListener('input', function(e) {
            let value = e.target.value.replace(/\D/g, '');
            if (value.length > 4) value = value.substr(0, 4);
            e.target.value = value;
        });
    }
    
    // Amount input validation for fee items
    document.addEventListener('input', function(e) {
        if (e.target.classList.contains('amount-input') && !e.target.disabled) {
            const value = parseFloat(e.target.value);
            const max = parseFloat(e.target.max);
            
            if (isNaN(value) || value < 0) {
                e.target.value = '0.00';
            } else if (value > max) {
                e.target.value = max.toFixed(2);
                showMessage('Cannot exceed original amount', 'warning');
            }
        }
    });
});
</script>

<style>
/* Professional Design Variables */
:root {
 --primary: #861d0bff;
  --primary-dark: #0cbe8bff;
    --primary-dark: #0ca323ff;
    --secondary: #7209b7;
    --success: #4cc9f0;
    --success-dark: #3a86ff;
    --danger: #f72585;
    --warning: #f8961e;
    --info: #06d6a0;
    --light: #f8f9fa;
    --dark: #212529;
    --gray: #6c757d;
    --gray-light: #e9ecef;
    --border: #dee2e6;
    --shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
    --shadow-lg: 0 10px 15px -3px rgba(0, 0, 0, 0.1), 0 4px 6px -2px rgba(0, 0, 0, 0.05);
    --shadow-xl: 0 20px 25px -5px rgba(0, 0, 0, 0.1), 0 10px 10px -5px rgba(0, 0, 0, 0.04);
    --radius: 12px;
    --radius-sm: 8px;
    --transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
}

/* Base Styles */
* {
    margin: 0;
    padding: 0;
    box-sizing: border-box;
}

body {
    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, sans-serif;
    background: #f8fafc;
    color: var(--dark);
    line-height: 1.6;
}

/* Page Hero */
.page-hero {
    background: linear-gradient(135deg, var(--primary) 0%, var(--primary-dark) 100%);
    color: white;
    padding: 40px 20px;
    margin-bottom: 30px;
    position: relative;
    overflow: hidden;
}

.hero-content {
    max-width: 1200px;
    margin: 0 auto;
    text-align: center;
}

.hero-content h1 {
    font-size: clamp(1.8rem, 4vw, 2.8rem);
    font-weight: 700;
    margin-bottom: 10px;
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 15px;
    flex-wrap: wrap;
}

.hero-subtitle {
    font-size: clamp(1rem, 2vw, 1.2rem);
    opacity: 0.9;
    margin-bottom: 30px;
    font-weight: 300;
}

/* Progress Steps */
.progress-steps {
    display: flex;
    justify-content: center;
    align-items: center;
    gap: clamp(20px, 4vw, 40px);
    margin-top: 30px;
    position: relative;
    flex-wrap: wrap;
}

.progress-steps::before {
    content: '';
    position: absolute;
    top: 20px;
    left: 50%;
    transform: translateX(-50%);
    width: 60%;
    height: 3px;
    background: rgba(255, 255, 255, 0.3);
    z-index: 1;
}

.step {
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 10px;
    z-index: 2;
    position: relative;
    min-width: 100px;
}

.step-icon {
    width: 44px;
    height: 44px;
    background: rgba(255, 255, 255, 0.2);
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-weight: 600;
    font-size: 1.1rem;
    transition: var(--transition);
}

.step-text {
    font-size: 0.9rem;
    font-weight: 500;
    opacity: 0.7;
    transition: var(--transition);
    text-align: center;
}

.step.active .step-icon {
    background: white;
    color: var(--primary);
    transform: scale(1.1);
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
}

.step.active .step-text {
    opacity: 1;
    font-weight: 600;
}

.step.completed .step-icon {
    background: var(--success);
    color: white;
}

/* Alerts */
.alert {
    display: flex;
    align-items: center;
    gap: 15px;
    padding: 16px 20px;
    border-radius: var(--radius-sm);
    margin-bottom: 30px;
    animation: slideIn 0.3s ease;
}

@keyframes slideIn {
    from {
        opacity: 0;
        transform: translateY(-10px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.alert-success {
    background: linear-gradient(135deg, #d1fae5 0%, #a7f3d0 100%);
    border-left: 4px solid var(--success);
    color: #065f46;
}

.alert-error {
    background: linear-gradient(135deg, #fee2e2 0%, #fecaca 100%);
    border-left: 4px solid var(--danger);
    color: #7f1d1d;
}

.alert-warning {
    background: linear-gradient(135deg, #fef3c7 0%, #fde68a 100%);
    border-left: 4px solid var(--warning);
    color: #92400e;
}

.alert-info {
    background: linear-gradient(135deg, #e0f2fe 0%, #bae6fd 100%);
    border-left: 4px solid var(--primary);
    color: #0369a1;
}

.alert i {
    font-size: 1.3rem;
}

.alert-content {
    flex: 1;
    font-size: 0.95rem;
}

.alert-content strong {
    display: block;
    margin-bottom: 4px;
}

.alert-close {
    background: none;
    border: none;
    color: inherit;
    cursor: pointer;
    opacity: 0.7;
    transition: var(--transition);
    padding: 5px;
    display: flex;
    align-items: center;
    justify-content: center;
}

.alert-close:hover {
    opacity: 1;
}

/* Dashboard Container */
.dashboard-container {
    max-width: 1200px;
    margin: 0 auto;
    padding: 0 20px;
}

/* Dashboard Card */
.dashboard-card {
    background: white;
    border-radius: var(--radius);
    box-shadow: var(--shadow-lg);
    overflow: hidden;
    margin-bottom: 30px;
    animation: fadeIn 0.5s ease;
}

@keyframes fadeIn {
    from {
        opacity: 0;
        transform: translateY(20px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.card-header {
    padding: 25px 30px;
    border-bottom: 1px solid var(--border);
    background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%);
}

.card-header h2 {
    margin: 0 0 8px 0;
    color: var(--dark);
    font-size: clamp(1.3rem, 3vw, 1.5rem);
    display: flex;
    align-items: center;
    gap: 12px;
    flex-wrap: wrap;
}

.card-subtitle {
    margin: 0;
    color: var(--gray);
    font-size: 0.95rem;
}

.card-body {
    padding: 30px;
}

@media (max-width: 768px) {
    .card-header,
    .card-body {
        padding: 20px;
    }
}

/* Search Section */
.search-section {
    margin-bottom: 0;
}

.search-section h3 {
    margin: 0 0 10px 0;
    color: var(--dark);
    font-size: 1.2rem;
    display: flex;
    align-items: center;
    gap: 10px;
}

.section-description {
    margin: 0 0 25px 0;
    color: var(--gray);
    font-size: 0.95rem;
}

/* Search Form */
.search-form {
    position: relative;
}

.search-container {
    position: relative;
}

.input-with-icon {
    position: relative;
}

.input-icon {
    position: absolute;
    left: 20px;
    top: 50%;
    transform: translateY(-50%);
    color: var(--gray);
    font-size: 1.1rem;
}

.search-input {
    width: 100%;
    padding: 16px 20px 16px 55px;
    border: 2px solid var(--border);
    border-radius: var(--radius-sm);
    font-size: 1rem;
    transition: var(--transition);
    background: white;
}

.search-input:focus {
    outline: none;
    border-color: var(--primary);
    box-shadow: 0 0 0 3px rgba(67, 97, 238, 0.1);
}

.input-hint {
    display: flex;
    align-items: center;
    gap: 8px;
    margin-top: 8px;
    color: var(--gray);
    font-size: 0.85rem;
    font-style: italic;
}

.input-hint i {
    color: var(--primary);
}

.search-results {
    position: absolute;
    top: 100%;
    left: 0;
    right: 0;
    background: white;
    border: 2px solid var(--border);
    border-top: none;
    border-radius: 0 0 var(--radius-sm) var(--radius-sm);
    max-height: 400px;
    overflow-y: auto;
    z-index: 1000;
    box-shadow: var(--shadow-lg);
    margin-top: -2px;
}

.search-loading {
    padding: 30px;
    text-align: center;
    color: var(--gray);
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 15px;
}

.search-no-results {
    padding: 30px;
    text-align: center;
    color: var(--gray);
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 15px;
    background: var(--light);
}

.search-no-results i {
    font-size: 2rem;
    color: var(--gray);
}

.search-results-list {
    padding: 10px 0;
}

.search-result-item {
    display: flex;
    align-items: center;
    gap: 15px;
    padding: 18px 20px;
    border-bottom: 1px solid var(--border);
    cursor: pointer;
    transition: var(--transition);
    background: white;
}

.search-result-item:hover {
    background: #f8fafc;
    transform: translateX(5px);
}

.search-result-item:last-child {
    border-bottom: none;
}

.result-avatar {
    width: 48px;
    height: 48px;
    background: linear-gradient(135deg, var(--primary-light) 0%, var(--primary) 100%);
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-size: 1.2rem;
    flex-shrink: 0;
}

.result-info {
    flex: 1;
    min-width: 0;
}

.result-name {
    font-weight: 600;
    color: var(--dark);
    margin-bottom: 6px;
    font-size: 1.05rem;
    overflow: hidden;
    text-overflow: ellipsis;
    white-space: nowrap;
}

.result-email, .result-phone, .result-id {
    font-size: 0.9rem;
    color: var(--gray);
    display: flex;
    align-items: center;
    gap: 8px;
    margin-bottom: 3px;
    overflow: hidden;
    text-overflow: ellipsis;
    white-space: nowrap;
}

.result-select {
    color: var(--primary);
    opacity: 0.7;
    transition: var(--transition);
    flex-shrink: 0;
}

.search-result-item:hover .result-select {
    opacity: 1;
    transform: translateX(5px);
}

/* Divider */
.divider {
    display: flex;
    align-items: center;
    margin: 40px 0 30px;
    position: relative;
}

.divider::before, .divider::after {
    content: '';
    flex: 1;
    height: 1px;
    background: var(--border);
}

.divider span {
    padding: 0 20px;
    color: var(--primary);
    font-weight: 600;
    font-size: 1rem;
    background: white;
    z-index: 1;
}

/* Register Prompt */
.register-prompt {
    display: flex;
    align-items: center;
    gap: 20px;
    background: linear-gradient(135deg, #f0f9ff 0%, #e0f2fe 100%);
    border: 1px solid var(--primary-light);
    border-radius: var(--radius-sm);
    padding: 25px;
    margin-bottom: 30px;
}

.prompt-icon {
    width: 60px;
    height: 60px;
    background: linear-gradient(135deg, var(--success) 0%, var(--success-dark) 100%);
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-size: 1.5rem;
    flex-shrink: 0;
}

.prompt-content {
    flex: 1;
    min-width: 0;
}

.prompt-content h4 {
    margin: 0 0 8px 0;
    color: var(--dark);
    font-size: 1.2rem;
}

.prompt-content p {
    margin: 0;
    color: var(--gray);
    font-size: 0.95rem;
    overflow: hidden;
    text-overflow: ellipsis;
}

/* Registration Form */
.form-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 20px;
    margin-bottom: 20px;
}

.form-group {
    margin-bottom: 25px;
}

.form-group label {
    display: flex;
    align-items: center;
    gap: 8px;
    margin-bottom: 10px;
    font-weight: 500;
    color: var(--dark);
    font-size: 0.95rem;
}

.form-group label i {
    color: var(--primary);
    width: 16px;
}

.required {
    color: var(--danger);
    font-weight: 600;
}

.form-control {
    width: 100%;
    padding: 14px 16px;
    border: 2px solid var(--border);
    border-radius: var(--radius-sm);
    font-size: 1rem;
    transition: var(--transition);
    background: white;
}

.form-control:focus {
    outline: none;
    border-color: var(--primary);
    box-shadow: 0 0 0 3px rgba(67, 97, 238, 0.1);
}

.input-group {
    display: flex;
}

.input-group-text {
    padding: 0 15px;
    background: var(--gray-light);
    border: 2px solid var(--border);
    border-right: none;
    border-radius: var(--radius-sm) 0 0 var(--radius-sm);
    display: flex;
    align-items: center;
    color: var(--gray);
    font-weight: 500;
}

.input-group .form-control {
    border-radius: 0 var(--radius-sm) var(--radius-sm) 0;
}

/* Buttons */
.btn {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    gap: 10px;
    padding: 14px 28px;
    border: none;
    border-radius: var(--radius-sm);
    font-size: 1rem;
    font-weight: 600;
    cursor: pointer;
    transition: var(--transition);
    text-decoration: none;
    text-align: center;
    white-space: nowrap;
}

.btn-primary {
    background: linear-gradient(135deg, var(--primary) 0%, var(--primary-dark) 100%);
    color: white;
    width: 100%;
    padding: 16px;
    font-size: 1.1rem;
}

.btn-primary:hover:not(:disabled) {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(67, 97, 238, 0.3);
}

.btn-success {
    background: linear-gradient(135deg, var(--success) 0%, var(--success-dark) 100%);
    color: white;
    width: 100%;
    padding: 16px;
    font-size: 1.1rem;
}

.btn-success:hover:not(:disabled) {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(76, 201, 240, 0.3);
}

.btn-change {
    display: inline-flex;
    align-items: center;
    gap: 8px;
    padding: 10px 20px;
    background: var(--gray-light);
    color: var(--dark);
    border-radius: var(--radius-sm);
    text-decoration: none;
    font-weight: 500;
    font-size: 0.9rem;
    transition: var(--transition);
    white-space: nowrap;
}

.btn-change:hover {
    background: var(--border);
    transform: translateY(-1px);
}

.btn-pay {
    width: 100%;
    padding: 20px;
    font-size: 1.1rem;
    border-radius: var(--radius);
    box-shadow: 0 4px 15px rgba(67, 97, 238, 0.2);
}

.btn-pay:hover:not(:disabled) {
    transform: translateY(-3px);
    box-shadow: 0 8px 25px rgba(67, 97, 238, 0.3);
}

.btn:disabled {
    opacity: 0.6;
    cursor: not-allowed;
    transform: none !important;
}

.form-actions {
    margin-top: 30px;
}

.security-note {
    text-align: center;
    margin-top: 15px;
    color: var(--gray);
    font-size: 0.9rem;
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 8px;
    flex-wrap: wrap;
}

.security-note i {
    color: var(--success);
}

/* Dashboard Grid */
.dashboard-grid {
    display: grid;
    grid-template-columns: 350px 1fr;
    gap: 30px;
    margin-bottom: 40px;
}

@media (max-width: 1024px) {
    .dashboard-grid {
        grid-template-columns: 1fr;
        gap: 20px;
    }
}

/* Student Card */
.student-card {
    background: white;
    border-radius: var(--radius);
    box-shadow: var(--shadow);
    overflow: hidden;
    height: fit-content;
}

.student-card .card-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 20px;
    flex-wrap: wrap;
    gap: 15px;
}

.student-card .card-header h3 {
    margin: 0;
    font-size: 1.2rem;
}

.student-info {
    padding: 25px;
}

.student-avatar {
    width: 80px;
    height: 80px;
    background: linear-gradient(135deg, var(--primary) 0%, var(--primary-dark) 100%);
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-size: 2rem;
    margin: 0 auto 25px;
}

.student-details {
    text-align: center;
}

.student-details h4 {
    margin: 0 0 20px 0;
    color: var(--dark);
    font-size: 1.3rem;
    word-break: break-word;
}

.detail-item {
    display: flex;
    align-items: center;
    gap: 12px;
    padding: 12px 0;
    border-bottom: 1px solid var(--border);
    text-align: left;
}

.detail-item:last-child {
    border-bottom: none;
}

.detail-item i {
    width: 20px;
    color: var(--primary);
    flex-shrink: 0;
}

.detail-item span {
    flex: 1;
    color: var(--gray);
    font-size: 0.95rem;
    word-break: break-word;
}

.registration-success {
    background: linear-gradient(135deg, #d1fae5 0%, #a7f3d0 100%);
    padding: 15px 20px;
    display: flex;
    align-items: center;
    gap: 12px;
    color: #065f46;
    font-weight: 500;
    border-top: 1px solid rgba(6, 95, 70, 0.1);
}

.registration-success i {
    font-size: 1.2rem;
    flex-shrink: 0;
}

.registration-success span {
    flex: 1;
    word-break: break-word;
}

/* Package Card */
.package-card {
    background: white;
    border-radius: var(--radius);
    box-shadow: var(--shadow);
    overflow: hidden;
}

.package-card .card-body {
    padding: 30px;
}

/* Select Wrapper */
.select-wrapper {
    position: relative;
}

.select-wrapper select {
    width: 100%;
    padding: 14px 45px 14px 16px;
    border: 2px solid var(--border);
    border-radius: var(--radius-sm);
    font-size: 1rem;
    appearance: none;
    background: white;
    cursor: pointer;
    transition: var(--transition);
}

.select-wrapper select:focus {
    outline: none;
    border-color: var(--primary);
    box-shadow: 0 0 0 3px rgba(67, 97, 238, 0.1);
}

.select-icon {
    position: absolute;
    right: 16px;
    top: 50%;
    transform: translateY(-50%);
    color: var(--gray);
    pointer-events: none;
}

.form-help {
    display: block;
    margin-top: 8px;
    color: var(--gray);
    font-size: 0.85rem;
}

/* Package Info Card */
.package-info-card {
    background: linear-gradient(135deg, #f0f9ff 0%, #e0f2fe 100%);
    border: 1px solid var(--primary-light);
    border-radius: var(--radius-sm);
    padding: 20px;
    margin: 20px 0;
}

.package-info-header {
    margin-bottom: 15px;
}

.package-info-header h4 {
    margin: 0;
    color: var(--dark);
    font-size: 1.1rem;
    display: flex;
    align-items: center;
    gap: 10px;
}

.package-info-body {
    display: grid;
    gap: 12px;
}

.info-row {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding-bottom: 8px;
    border-bottom: 1px solid rgba(67, 97, 238, 0.1);
}

.info-row:last-child {
    border-bottom: none;
    padding-bottom: 0;
}

.info-label {
    font-weight: 500;
    color: var(--gray);
}

.info-value {
    color: var(--dark);
    font-weight: 500;
}

.info-value.amount {
    font-weight: 700;
    color: var(--primary);
}

/* Fee Items */
.section-header {
    margin-bottom: 25px;
    padding-bottom: 15px;
    border-bottom: 1px solid var(--border);
}

.section-header h4 {
    margin: 0 0 8px 0;
    color: var(--dark);
    font-size: 1.1rem;
    display: flex;
    align-items: center;
    gap: 10px;
}

.fee-items-grid {
    display: grid;
    gap: 15px;
    margin-bottom: 30px;
}

.fee-item-card {
    background: white;
    border: 2px solid var(--border);
    border-radius: var(--radius-sm);
    padding: 20px;
    display: flex;
    justify-content: space-between;
    align-items: center;
    transition: var(--transition);
    flex-wrap: wrap;
    gap: 20px;
}

.fee-item-card:hover {
    border-color: var(--primary-light);
    box-shadow: 0 4px 12px rgba(67, 97, 238, 0.1);
}

.fee-item-card.selected {
    border-color: var(--primary);
    background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%);
}

.fee-item-header {
    display: flex;
    align-items: flex-start;
    gap: 15px;
    flex: 1;
    min-width: 0;
}

.custom-checkbox {
    position: relative;
    flex-shrink: 0;
}

.custom-checkbox input[type="checkbox"] {
    position: absolute;
    opacity: 0;
    cursor: pointer;
    height: 0;
    width: 0;
}

.checkbox-box {
    width: 24px;
    height: 24px;
    border: 2px solid var(--border);
    border-radius: 6px;
    display: flex;
    align-items: center;
    justify-content: center;
    cursor: pointer;
    transition: var(--transition);
    background: white;
    margin-top: 2px;
}

.checkbox-box i {
    color: white;
    font-size: 0.8rem;
    opacity: 0;
    transition: var(--transition);
}

.custom-checkbox input:checked + label .checkbox-box {
    background: var(--primary);
    border-color: var(--primary);
}

.custom-checkbox input:checked + label .checkbox-box i {
    opacity: 1;
}

.fee-item-info {
    flex: 1;
    min-width: 0;
}

.fee-item-info h5 {
    margin: 0 0 8px 0;
    color: var(--dark);
    font-size: 1rem;
    font-weight: 600;
    word-break: break-word;
}

.fee-description {
    margin: 0 0 8px 0;
    color: var(--gray);
    font-size: 0.9rem;
    line-height: 1.4;
    word-break: break-word;
}

.fee-item-meta {
    display: flex;
    align-items: center;
    gap: 12px;
    flex-wrap: wrap;
}

.fee-code {
    color: var(--gray);
    font-size: 0.85rem;
    display: flex;
    align-items: center;
    gap: 5px;
}

.refundable-badge {
    background: linear-gradient(135deg, #d1fae5 0%, #a7f3d0 100%);
    color: #065f46;
    padding: 4px 10px;
    border-radius: 20px;
    font-size: 0.75rem;
    font-weight: 500;
    display: flex;
    align-items: center;
    gap: 5px;
}

.fee-item-amount {
    text-align: right;
    flex-shrink: 0;
}

.amount-control {
    display: flex;
    flex-direction: column;
    align-items: flex-end;
    gap: 5px;
}

.amount-input-group {
    display: flex;
    align-items: center;
    gap: 8px;
}

.currency-label {
    font-weight: 600;
    color: var(--dark);
    font-size: 1rem;
}

.amount-input {
    width: 120px;
    padding: 10px 12px;
    border: 2px solid var(--border);
    border-radius: var(--radius-sm);
    font-size: 1rem;
    text-align: right;
    font-weight: 500;
    transition: var(--transition);
    background: white;
}

.amount-input:focus {
    outline: none;
    border-color: var(--primary);
    box-shadow: 0 0 0 3px rgba(67, 97, 238, 0.1);
}

.amount-input:disabled {
    background: var(--gray-light);
    color: var(--gray);
    cursor: not-allowed;
}

.amount-hint small {
    color: var(--gray);
    font-size: 0.8rem;
    font-style: italic;
}

.no-items {
    text-align: center;
    padding: 60px 20px;
    color: var(--gray);
    background: var(--gray-light);
    border-radius: var(--radius-sm);
    border: 2px dashed var(--border);
}

.no-items i {
    font-size: 3rem;
    margin-bottom: 15px;
    opacity: 0.5;
}

/* Selected Summary */
.selected-summary {
    background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%);
    border-radius: var(--radius-sm);
    padding: 20px;
    margin-bottom: 20px;
    border: 1px solid var(--border);
}

.selected-summary .summary-header {
    margin-bottom: 15px;
    padding-bottom: 10px;
    border-bottom: 1px solid var(--border);
}

.selected-summary .summary-header h4 {
    margin: 0;
    font-size: 1.1rem;
    color: var(--dark);
    display: flex;
    align-items: center;
    gap: 10px;
}

.selected-items-list {
    display: grid;
    gap: 12px;
}

.selected-item {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 12px;
    background: white;
    border-radius: 6px;
    border: 1px solid var(--border-light);
}

.selected-item-info {
    display: flex;
    justify-content: space-between;
    align-items: center;
    width: 100%;
    gap: 15px;
}

.selected-item-name {
    color: var(--dark);
    font-weight: 500;
    flex: 1;
    word-break: break-word;
}

.selected-item-amount {
    color: var(--primary);
    font-weight: 600;
    font-size: 1rem;
    flex-shrink: 0;
}

/* Payment Summary */
.payment-summary {
    background: linear-gradient(135deg, var(--primary) 0%, var(--primary-dark) 100%);
    color: white;
    border-radius: var(--radius);
    padding: 25px;
    margin: 30px 0;
    border: 1px solid var(--primary);
}

.payment-summary .summary-header {
    margin-bottom: 20px;
    border-bottom: 1px solid rgba(255, 255, 255, 0.2);
    padding-bottom: 15px;
}

.payment-summary .summary-header h4 {
    margin: 0;
    font-size: 1.1rem;
    display: flex;
    align-items: center;
    gap: 10px;
    color: white;
}

.summary-details {
    display: grid;
    gap: 15px;
}

.summary-row {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding-bottom: 12px;
    border-bottom: 1px solid rgba(255, 255, 255, 0.1);
}

.summary-row:last-child {
    border-bottom: none;
}

.summary-row.total {
    font-size: 1.5rem;
    font-weight: 700;
    padding-top: 15px;
    border-top: 2px solid rgba(255, 255, 255, 0.2);
    border-bottom: none;
}

/* Payment Methods */
.payment-methods {
    margin-bottom: 30px;
}

.payment-methods h4 {
    margin: 0 0 20px 0;
    color: var(--dark);
    font-size: 1.1rem;
    display: flex;
    align-items: center;
    gap: 10px;
}

.method-options {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 15px;
}

.method-option {
    display: flex;
    align-items: center;
    justify-content: space-between;
    padding: 20px;
    border: 2px solid var(--border);
    border-radius: var(--radius-sm);
    cursor: pointer;
    transition: var(--transition);
    background: white;
    position: relative;
}

.method-option input[type="radio"] {
    position: absolute;
    opacity: 0;
}

.method-option.active,
.method-option.selected {
    border-color: var(--primary);
    background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%);
}

.method-content {
    display: flex;
    align-items: center;
    gap: 15px;
    flex: 1;
    min-width: 0;
}

.method-icon {
    width: 48px;
    height: 48px;
    background: linear-gradient(135deg, var(--primary) 0%, var(--primary-dark) 100%);
    border-radius: 12px;
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-size: 1.3rem;
    flex-shrink: 0;
}

.method-info {
    flex: 1;
    min-width: 0;
}

.method-info h5 {
    margin: 0 0 5px 0;
    color: var(--dark);
    font-size: 1rem;
    word-break: break-word;
}

.method-info p {
    margin: 0;
    color: var(--gray);
    font-size: 0.9rem;
    word-break: break-word;
}

.method-check {
    color: var(--primary);
    font-size: 1.2rem;
    opacity: 0;
    transition: var(--transition);
    flex-shrink: 0;
}

.method-option.active .method-check,
.method-option.selected .method-check {
    opacity: 1;
}

/* Quick Actions */
.quick-actions {
    margin-top: 50px;
}

.quick-actions h3 {
    margin: 0 0 25px 0;
    color: var(--dark);
    font-size: 1.3rem;
    display: flex;
    align-items: center;
    gap: 12px;
}

.actions-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 20px;
}

.action-card {
    display: flex;
    align-items: center;
    gap: 20px;
    padding: 20px;
    background: white;
    border-radius: var(--radius);
    box-shadow: var(--shadow);
    text-decoration: none;
    color: var(--dark);
    transition: var(--transition);
    border: 1px solid var(--border);
}

.action-card:hover {
    transform: translateY(-5px);
    box-shadow: var(--shadow-lg);
    border-color: var(--primary-light);
}

.action-icon {
    width: 56px;
    height: 56px;
    border-radius: 12px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.5rem;
    color: white;
    flex-shrink: 0;
}

.action-icon.test {
    background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%);
}

.action-icon.quick {
    background: linear-gradient(135deg, #10b981 0%, #059669 100%);
}

.action-icon.history {
    background: linear-gradient(135deg, #8b5cf6 0%, #7c3aed 100%);
}

.action-icon.manage {
    background: linear-gradient(135deg, #ec4899 0%, #db2777 100%);
}

.action-info {
    flex: 1;
    min-width: 0;
}

.action-info h4 {
    margin: 0 0 6px 0;
    font-size: 1.1rem;
    word-break: break-word;
}

.action-info p {
    margin: 0;
    color: var(--gray);
    font-size: 0.9rem;
    word-break: break-word;
}

.action-arrow {
    color: var(--primary);
    opacity: 0.7;
    transition: var(--transition);
    flex-shrink: 0;
}

.action-card:hover .action-arrow {
    opacity: 1;
    transform: translateX(5px);
}

/* Animation for fee items section */
#fee-items-section {
    opacity: 0;
    transform: translateY(20px);
    transition: opacity 0.5s ease, transform 0.5s ease;
}

/* Responsive Design for Mobile */
@media (max-width: 768px) {
    .page-hero {
        padding: 30px 15px;
    }
    
    .progress-steps::before {
        display: none;
    }
    
    .progress-steps {
        gap: 20px;
    }
    
    .step {
        min-width: 80px;
    }
    
    .step-icon {
        width: 36px;
        height: 36px;
        font-size: 0.9rem;
    }
    
    .step-text {
        font-size: 0.8rem;
    }
    
    .alert {
        padding: 12px 15px;
        flex-wrap: wrap;
        gap: 10px;
    }
    
    .alert-content {
        font-size: 0.9rem;
    }
    
    .dashboard-container {
        padding: 0 15px;
    }
    
    .form-grid {
        grid-template-columns: 1fr;
        gap: 15px;
    }
    
    .search-results {
        position: static;
        margin-top: 10px;
        border: 2px solid var(--border);
        border-radius: var(--radius-sm);
    }
    
    .register-prompt {
        flex-direction: column;
        text-align: center;
        padding: 20px;
        gap: 15px;
    }
    
    .prompt-icon {
        width: 50px;
        height: 50px;
        font-size: 1.3rem;
    }
    
    .fee-item-card {
        flex-direction: column;
        align-items: stretch;
        gap: 20px;
    }
    
    .fee-item-amount {
        text-align: left;
        width: 100%;
    }
    
    .amount-control {
        align-items: stretch;
    }
    
    .amount-input-group {
        width: 100%;
    }
    
    .amount-input {
        width: 100%;
    }
    
    .selected-item-info {
        flex-direction: column;
        align-items: flex-start;
        gap: 5px;
    }
    
    .method-options {
        grid-template-columns: 1fr;
    }
    
    .actions-grid {
        grid-template-columns: 1fr;
    }
    
    .btn {
        padding: 12px 20px;
        font-size: 0.95rem;
    }
    
    .btn-pay {
        padding: 16px;
        font-size: 1rem;
    }
}

/* Extra Small Devices */
@media (max-width: 480px) {
    .card-header {
        flex-direction: column;
        align-items: flex-start;
        gap: 15px;
    }
    
    .card-header h3 {
        font-size: 1.1rem;
    }
    
    .btn-change {
        width: 100%;
        justify-content: center;
    }
    
    .student-avatar {
        width: 60px;
        height: 60px;
        font-size: 1.5rem;
    }
    
    .student-details h4 {
        font-size: 1.1rem;
    }
    
    .detail-item {
        font-size: 0.9rem;
    }
    
    .package-card .card-body {
        padding: 20px;
    }
    
    .payment-summary {
        padding: 20px;
    }
    
    .summary-row.total {
        font-size: 1.3rem;
    }
    
    .method-content {
        gap: 10px;
    }
    
    .method-icon {
        width: 40px;
        height: 40px;
        font-size: 1.1rem;
    }
    
    .method-info h5 {
        font-size: 0.95rem;
    }
    
    .method-info p {
        font-size: 0.85rem;
    }
    
    .action-card {
        padding: 15px;
        gap: 15px;
    }
    
    .action-icon {
        width: 48px;
        height: 48px;
        font-size: 1.3rem;
    }
    
    .action-info h4 {
        font-size: 1rem;
    }
    
    .action-info p {
        font-size: 0.85rem;
    }
}

/* Print Styles */
@media print {
    .page-hero,
    .quick-actions,
    .btn-change,
    .alert-close,
    .security-note {
        display: none;
    }
    
    .dashboard-card,
    .student-card,
    .package-card {
        box-shadow: none;
        border: 1px solid #ddd;
    }
    
    .btn {
        display: none;
    }
    
    .form-group select,
    .form-group input:not([type="checkbox"]):not([type="radio"]) {
        border: 1px solid #ddd;
        background: white;
    }
}
</style>